;stepmtre.asm; fifth stepper motor program to get basic functionality.
;uses PIC 16F628 with internal 4 Mhz clock, MCLRE cleared, and FOSC = 100b.
;calibrate step values into EEPROM.
;john waller 2004oct13.
;
;
;
;
__config H'3F10' ;osc = INT (no xtal), wdt = OFF, por = ON (PIC16F627/628),
		 ;MCLRE cleared, FOSC = 100b, A6 and A7 as digital I/O.

#DEFINE BLOCK0   bcf $03,7   	;STATUS bit 7.
#DEFINE BLOCK1   bsf $03,7   	;STATUS bit 7.
#DEFINE PAGE0S1  %00000001   	;Page 0, sub-page 1.
#DEFINE PAGE1S0  %00001000   	;Page 1, sub-page 0.
#DEFINE PAGE2S0  %00010000	;Page 2, sub-page 0.
#DEFINE RP0CLR   bcf $03,5   	;STATUS bit 5.
#DEFINE RP0SET   bsf $03,5   	;STATUS bit 5.
#DEFINE RP1CLR   bcf $03,6   	;STATUS bit 6.
#DEFINE RP1SET   bsf $03,6   	;STATUS bit 6.

INDF:    .EQU $00	;Indirect address, bank 0, 1, 2, 3.
TMRO:	 .EQU $01	;Timer 0 value, bank 0, 2.
OPTION:  .EQU $01	;Option register, bank 1, 3.
PCL:     .EQU $02	;Program counter low byte, bank 0, 1, 2, 3.
STATUS:  .EQU $03	;Status flags, bank 0, 1, 2, 3.
FSR:     .EQU $04	;Indirect address pointer, bank 0, 1, 2, 3.
PORTA:   .EQU $05	;Port A, bank 0.
TRISA:   .EQU $05	;Port A direction register, bank 1.
PORTB:   .EQU $06	;Port B, bank 0, 2.
TRISB:   .EQU $06	;Port B direction register, bank 1, 3.
PCLATH:	 .EQU $0A	;Program counter high byte setting register, bank 0, 1, 2, 3.
INTCON:  .EQU $0B	;Interrupt control, bank 0, 1, 2, 3.
PIR1:	 .EQU $0C	;Peripheral interrupt flags, bank 0.
PIE1:    .EQU $0C	;Peripheral interrupt enable, bank 1.
TMR1L:	 .EQU $0E	;Timer 1 low byte, bank 0.
PCON:	 .EQU $0E	;Internal oscillator frequency??, POR, BOD, bank 1
TMR1H:	 .EQU $0F	;Timer 1 high byte, bank 0.
T1CON:	 .EQU $10	;Timer 1 control, bank 0.
TMR2:	 .EQU $11	;Timer 2 byte, bank 0.
T2CON:	 .EQU $12	;Timer 2 control, bank 0.
PR2:	 .EQU $12	;Timer 2 period register, bank 1.
CCPR1L:	 .EQU $15	;Capture/Compare/PWM low byte, bank 0.
CCPR1H:	 .EQU $16	;Capture/Compare/PWM high byte, bank 0.
CCP1CON: .EQU $17	;Capture/compare/PWM control, bank 0.
RCSTA:	 .EQU $18	;receive status and control register, bank 0.
TXSTA:	 .EQU $18	;transmit status and control register, bank 1.
TXREG:	 .EQU $19	;USART transmit register, bank 0.
SPBRG:	 .EQU $19	;baud rate factor, bank 1.
RCREG:	 .EQU $1A	;USART receive register, bank 0.
EEDATA:	 .EQU $1A	;EEPROM data, bank 1.
EEADR:   .EQU $1B	;EEPROM address, bank 1.
EECON1:  .EQU $1C	;EEPROM control, bank 1.
EECON2:  .EQU $1D	;EEPROM control, bank 1.
CMCON:	 .EQU $1F	;Comparator control register, bank 0.

;........GENERAL PURPOSE REGISTERS

;........BANK 0

TMR1HS:	.EQU $20	;Timer 1 high byte stored.
CNTAH:	.EQU $21	;Step counter 1 (A) high.
CNTAL:  .EQU $22	;Step counter 1 (A) low.
COUNTA: .EQU $23	;Stepper counter 1.
CNTBH:	.EQU $24	;Step counter 2 (B) high.
CNTBL:  .EQU $25	;Step counter 2 (B) low.
COUNTB: .EQU $26	;Stepper counter 2.
TABOUT: .EQU $27	;Temporary store for table output.
LMPCNT: .EQU $28	;Count down for lamp toggle.
LMPOUT: .EQU $29	;Temporary store for lamp bits.
BSYFLA:	.EQU $2A	;Busy flags A; bit 0, initialise/kill; bit 1, set red;
			;bit 2, set yellow; bit 3, set green; bit 4, step CW.
BSYFLB:	.EQU $2B	;Busy flags B; bit 0, initialise/kill; bit 1, set red;
			;bit 2, set yellow; bit 3, set green; bit 4, step CW.
SIGSTA:	.EQU $2C	;signal state 1 (A); bit 0, none; bit 1, red; bit 2, yellow;
			;bit 3, green; bits 4..7, unassigned.
SIGSTB:	.EQU $2D	;signal state 2 (B); bit 0, none; bit 1, red; bit 2, yellow;
			;bit 3, green; bits 4..7, unassigned.
LOOPA:	.EQU $2E	;General purpose counter.
EEPROD:	.EQU $2F	;Temporary store for data from EEPROM.
CALFLG:	.EQU $30	;Calibration flags.
CALCTH:	.EQU $31	;Calibration count high.
CALCTL:	.EQU $32	;Calibration count low.
CALADD:	.EQU $33	;Calibration address in EEPROM / 2.
			
			;Values for step counts as read from EEPROM, and written to EEPROM
			;during calibration.
CTGRAH:	.EQU $40	;High count (A) green to red.
CTGRAL:	.EQU $41	;Low count (A) green to red.
CTGRBH:	.EQU $42	;High count (B) green to red.
CTGRBL:	.EQU $43	;Low count (B) green to red.
CTGYAH:	.EQU $44	;High count (A) green to yellow.
CTGYAL:	.EQU $45	;Low count (A) green to yellow.
CTGYBH:	.EQU $46	;High count (B) green to yellow.
CTGYBL:	.EQU $47	;Low count (B) green to yellow.
CTRGAH:	.EQU $48	;High count (A) red to green.
CTRGAL:	.EQU $49	;Low count (A) red to green.
CTRGBH:	.EQU $4A	;High count (B) red to green.
CTRGBL:	.EQU $4B	;Low count (B) red to green.
CTRYAH:	.EQU $4C	;High count (A) red to yellow.
CTRYAL:	.EQU $4D	;Low count (A) red to yellow.
CTRYBH:	.EQU $4E	;High count (B) red to yellow.
CTRYBL:	.EQU $4F	;Low count (B) red to yellow.
CTYGAH:	.EQU $50	;High count (A) yellow to green.
CTYGAL:	.EQU $51	;Low count (A) yellow to green.
CTYGBH:	.EQU $52	;High count (B) yellow to green.
CTYGBL:	.EQU $53	;Low count (B) yellow to green.
CTYRAH:	.EQU $54	;High count (A) yellow to red.
CTYRAL:	.EQU $55	;Low count (A) yellow to red.
CTYRBH:	.EQU $56	;High count (B) yellow to red.
CTYRBL:	.EQU $57	;Low count (B) yellow to red.

;........CONSTANTS

LCINIT:	.EQU 64		;Count down init for lamp toggle.
NUMSTV:	.EQU 24		;Number of step count values.
T1MASK: .EQU %00000011	;Timer 1 background loop mask; for each "zero" divide the loop time
			;by a factor of two.
T2MASK:	.EQU %00111111	;Timer 1 calibrate loop mask.

;........BIT VALUES

C:      .EQU 0          ;Carry status.
CW:	.EQU 4		;Clockwise stepping (required when moving to yellow).
F:      .EQU 1          ;Select file register.
G:	.EQU 3		;Green state.
GIE:	.EQU 7		;General interrupt enable.
IK:	.EQU 0		;Init/kill busy flag.
R:	.EQU 1		;Red state.
RD:	.EQU 0		;EEPROM read control.
SG:	.EQU 3		;Set green busy flag.
SR:	.EQU 1		;Set red busy flag.
SY:	.EQU 2		;Set yellow busy flag.
W:      .EQU 0          ;Select working register.
WR:	.EQU 1		;EEPROM write control initiate.
WREN:	.EQU 2		;EEPROM write enable.
WRERR:	.EQU 3		;EEPROM write error.
Y:	.EQU 2		;Yellow state.
Z:      .EQU 2          ;Zero status.

;..........

        .ORG $0000
        goto INITIALISE		;Setup everything.
        
        .ORG $0004		;Location for interrupt vector, if used.
        
;..........
		;page 0, sub-page 0 tables: make sure they don't overrun.
		
;.........

		;Select CW or CCW stepping for calibration. Destination address
		;for EEPROM / 2 in W. Returns 0 for ACW, %10 for CW.

SCWACW:	addwf PCL,F		;Add address to program counter.
	retlw %10		;CW
	retlw %10		;CW
	retlw %10		;CW
	retlw %10		;CW
	retlw 0			;ACW
	retlw 0			;ACW
	retlw 0			;ACW
	retlw 0			;ACW
	retlw 0			;ACW
	retlw 0			;ACW
	retlw %10		;CW
	retlw %10		;CW
	
;..........

		;Wave control bits table for wave operation, entered with count in W.
		;
WAVE:	andlw %00000011		;Mask in bits of count and add
	addwf PCL,F		;to program counter.
	retlw %00000001		;Return
	retlw %00000010		;appropriate
	retlw %00000100		;bit to
	retlw %00001000		;energise.

;..........

		;Background loop is continuous.
		;Entered from INITIALISE.
		;Calls.............; calls stack depth ??
BACKGROUND:	nop		;Do stuff here prior to background loop.
BACKG1:	btfsc TMR1HS,0		;Skip if timer 1 flag is cleared.
	goto BACKG3		;
	movf TMR1H,W		;Get timer 1 value,
	andlw T1MASK		;mask out bits to be ignored,	
	btfss STATUS,Z		;and skip if result zero.
	goto BACKG1
	bsf TMR1HS,0		;Set timer 1 flag to inhibit false zero indications.
	call STPOFF		;Turn off stepper motors.
	btfsc PORTA,5		;Skip if RUN flag cleared and do initialise/KILL stuff.
	goto BACKG2		;
	call TOGLPS		;Toggle both lamps.
	call DOINITKILLSIGA	;Do initialise/KILL for signal A.
	call DOINITKILLSIGB	;Do initialise/KILL for signal B.
	goto BACKG1		;
BACKG2:	bsf PORTA,6		;Turn on
	bsf PORTA,7		;both lamps.
	call DORUNSIGA		;Do RUN for signal A.
	call DORUNSIGB		;Do RUN for signal B.
	goto BACKG1		;
BACKG3:	movf TMR1H,W		;Get timer 1 value,
	andlw T1MASK		;mask out bits to be ignored,	
	btfsc STATUS,Z		;and skip if result not zero.
	goto BACKG1		;
	bcf TMR1HS,0		;Reset timer 1 flag so's next timer zero is recognised.
	goto BACKG1		;
	
;..........

CALIBRATE: nop	;Does a calibration step, if invoked, else does nothing.

	btfsc CALFLG,0		;Skip and return
	goto CALIB1		;if calibrate
	btfss PORTA,5		;flag and "RUN"
	return			;both clear.
	call GETCALADD		;Get calibrate
	movwf CALADD		;address / 2
	addlw 244		;and return
	btfsc STATUS,C		;if out of
	return			;range.
	bsf CALFLG,0		;Set calibrate flag.
	movfw CALADD		;Get
	call SCWACW		;step
	iorwf CALFLG,F		;direction.
CALIB1:	btfsc PORTA,5		;Skip if "RUN" cleared.
	goto CALIB2		;
				;Stop stepping, write count to EEPROM, and clear flags
				;and counters.
	bcf STATUS,C		;Clear carry and multiply
	rlf CALADD,W		;address by two.
	RP0SET			;Bank 1.
	movwf EEADR		;Set EEPROM address.
	RP0CLR			;Bank 0.
	movfw CALCTH		;Get high count byte.
	RP0SET			;Bank 1.
	movwf EEDATA		;Set EEPROM data.
	RP0CLR			;Bank 0.
	call WRITEEEPROM	;Write to EEPROM.
	RP0SET			;Bank 1.
	incf EEADR,F		;Increment EEPROM address.
	RP0CLR			;Bank 0.
	movfw CALCTL		;Get low count byte.
	RP0SET			;Bank 1.
	movwf EEDATA		;Set EEPROM data.
	RP0CLR			;Bank 0.
	call WRITEEEPROM	;Write to EEPROM.
	clrf CALCTH		;Clear
	clrf CALCTL		;count.
	clrf CALFLG		;Clear calibrate flags
	bsf CALFLG,7		;except bit 7.
	return			;
				;Do one step in the appropriate direction and output,
				;and increment counters.
CALIB2:	btfsc CALFLG,1		;Skip if direction ACW.
	goto CALIB4		;
	btfsc CALADD,0		;Skip if output is A.
	goto CALIB3		;
	call STACWA		;Step A ACW.
	goto CALIB6		;
CALIB3:	call STACWB		;Step B ACW.
	goto CALIB6		;
CALIB4:	btfsc CALADD,0		;Skip if output is A.
	goto CALIB5		;
	call STCWA		;Step A CW.
	goto CALIB6		;
CALIB5:	call STCWB		;Step B CW.
	goto CALIB6		;
CALIB6:	call INCNTC		;Increment counters.
	return			;
	
;..........

CALIBRATELOOP:	nop
		;Calibrate loop operates similarly to the background loop, but with much
		;slower steps. Exit from the loop can only be made with power off, since
		;MCLRE is cleared.
		
CALLP1:	btfsc TMR1HS,0		;Skip if timer 1 flag is cleared.
	goto CALLP3		;
	movf TMR1H,W		;Get timer 1 value,
	andlw T2MASK		;mask out bits to be ignored,	
	btfss STATUS,Z		;and skip if result zero.
	goto CALLP1
	bsf TMR1HS,0		;Set timer 1 flag to inhibit false zero indications.
	call STPOFF		;Turn off stepper motors.
	btfsc CALFLG,7		;Skip if "RUN" not yet cleared.
	goto CALLP2		;Else start calibration.
	btfsc PORTA,5		;Skip if "RUN" input cleared.
	goto CALLP1		;
	bsf CALFLG,7		;Set "RUN" cleared flag.
	goto CALLP1		;
				;Calibration starts here.
CALLP2:	call CALIBRATE		;Go calibrate.
	goto CALLP1		;
CALLP3:	movf TMR1H,W		;Get timer 1 value,
	andlw T2MASK		;mask out bits to be ignored,	
	btfsc STATUS,Z		;and skip if result not zero.
	goto CALLP1		;
	bcf TMR1HS,0		;Reset timer 1 flag so's next timer zero is recognised.
	goto CALLP1		;
	
;..........

		;Decrement the A (output 1) high and low counters, returning one if count
		;incomplete, else zero.
DECNTA:	decfsz CNTAL,F		;Skip if decremented low value zero.
	retlw 1			;Return not done.
	movf CNTAH,F		;Skip if high value
	btfsc STATUS,Z		;not zero.
	retlw 0			;Return done.
	decf CNTAH,F		;Decrement high value.
	movlw 255		;reset low value
	movwf CNTAL		;to maximum.
	retlw 1			;Return not done.
	
;..........

		;Decrement the B (output 2) high and low counters, returning one if count
		;incomplete, else zero.
DECNTB:	decfsz CNTBL,F		;Skip if decremented low value zero.
	retlw 1			;Return not done.
	movf CNTBH,F		;Skip if high value
	btfsc STATUS,Z		;not zero.
	retlw 0			;Return done.
	decf CNTBH,F		;Decrement high value.
	movlw 255		;reset low value
	movwf CNTBL		;to maximum.
	retlw 1			;Return not done.
	
;..........

DOINITKILLSIGA:	nop
		;Highlevel routine for initialise and kill for signal A.
	btfss BSYFLA,IK		;Skip if initialise/KILL busy.
	goto DOIKA2		;
	call STACWA		;Step A ACW and
	call DECNTA		;decrement counter.
	addlw 255		;Test and skip
	btfss STATUS,Z		;if not done.
	goto DOIKA1		;Else stop stepping.
	return			;
DOIKA1:	clrf BSYFLA		;Clear busy flags
	clrf SIGSTA		;and set signal
	bsf SIGSTA,G		;state to green.
	return			;
DOIKA2:	btfsc SIGSTA,G		;Skip if not in green state.
	return			;
	movfw CTRGAH		;Set red
	movwf CNTAH		;to green
	movfw CTRGAL		;counter
	movwf CNTAL		;values.
	bsf BSYFLA,IK		;Set busy flag.
	return			;
	
;..........

DOINITKILLSIGB:	nop
		;Highlevel routine for initialise and kill for signal B.
	btfss BSYFLB,IK		;Skip if initialise/KILL busy.
	goto DOIKB2		;
	call STACWB		;Step B ACW and
	call DECNTB		;decrement counter.
	addlw 255		;Test and skip
	btfss STATUS,Z		;if not done.
	goto DOIKB1		;Else stop stepping.
	return			;
DOIKB1:	clrf BSYFLB		;Clear busy flags
	clrf SIGSTB		;and set signal
	bsf SIGSTB,G		;state to green.
	return			;
DOIKB2:	btfsc SIGSTB,G		;Skip if not in green state.
	return			;
	movfw CTRGBH		;Set red
	movwf CNTBH		;to green
	movfw CTRGBL		;counter
	movwf CNTBL		;values.
	bsf BSYFLB,IK		;Set busy flag.
	return			;
	
;..........

DORUNSIGA:	nop
		;Highlevel routine for run state for signal A.
				;Do a switch-case for the signal colour required.
				;Do nothing unless colour is R, Y, or G.
				;Continue stepping if signal busy.
	movf BSYFLA,F		;Skip if any bits
	btfsc STATUS,Z		;set in busy flag.
	goto DORUA1		;
	btfsc BSYFLA,SG		;Skip if not set green busy,
	call SETA2G		;else continue stepping.
	btfsc BSYFLA,SR		;Skip if not set red busy,
	call SETA2R		;else continue stepping.
	btfsc BSYFLA,SY		;Skip if not set yellow busy,
	call SETA2Y		;else continue stepping.
	return			;
DORUA1:	swapf PORTB,W		;Mask in port B
	andlw %00000011		;bits 4 and 5.
	btfss STATUS,Z		;Skip if case zero (do nothing).
	goto DORUA2		;
	return			;
DORUA2:	addlw 255		;Skip if
	btfss STATUS,Z		;case one.
	goto DORUA3		;
	call SETA2R		;Handle set signal A to red.
	return			;
DORUA3:	addlw 255		;Skip if
	btfss STATUS,Z		;case two.
	goto DORUA4		;
	call SETA2G		;Handle set signal A to green.
	return			;
DORUA4:	addlw 255		;Skip if
	btfss STATUS,Z		;case three.
	return			;Else do nothing.
	call SETA2Y		;Handle set signal A yellow.
	return			;
	
;..........

DORUNSIGB:	nop
		;Highlevel routine for run state for signal B.
				;Do a switch-case for the signal colour required.
				;Do nothing unless colour is R, Y, or G.
				;Continue stepping if signal busy.
	movf BSYFLB,F		;Skip if any bits
	btfsc STATUS,Z		;set in busy flag.
	goto DORUB1		;
	btfsc BSYFLB,SG		;Skip if not set green busy,
	call SETB2G		;else continue stepping.
	btfsc BSYFLB,SR		;Skip if not set red busy,
	call SETB2R		;else continue stepping.
	btfsc BSYFLB,SY		;Skip if not set yellow busy,
	call SETB2Y		;else continue stepping.
	return			;
DORUB1:	swapf PORTB,W		;Mask in port B
	andlw %00001100		;bits 6 and 7.
	btfss STATUS,Z		;Skip if case zero (do nothing).
	goto DORUB2		;
	return			;
DORUB2:	addlw 252		;Skip if
	btfss STATUS,Z		;case four.
	goto DORUB3		;
	call SETB2R		;Handle set signal B to red.
	return			;
DORUB3:	addlw 252		;Skip if
	btfss STATUS,Z		;case eight.
	goto DORUB4		;
	call SETB2G		;Handle set signal B to green.
	return			;
DORUB4:	addlw 252		;Skip if
	btfss STATUS,Z		;case twelve.
	return			;Else do nothing.
	call SETB2Y		;Handle set signal B yellow.
	return			;
	
;..........

GETCALADD: nop	;Gets the address / 2 from the switches and returns it in W.
	swapf PORTB,W		;Get high nibb to low nibb.
	andlw %00001111		;Mask in new lower nibb.
	return			;
	
;..........

		;Increments high and low counters for calibration.
INCNTC:	incf CALCTL,F		;Increment low count.
	btfss STATUS,Z		;Skip if zero.
	return			;
	incf CALCTH,F		;
	return			;
	
;..........

		;Initialising ports and others. Entered at startup.
		;Called from none; calls stack depth ??.
INITIALISE:	bsf PCON,3	;Internal oscillator to 4 MHz; has no effect???
	clrf STATUS		;Bank and block 0 and clear all flags.
				;Clear all gpr locations in banks 0, 1, 2, 3.
	movlw $20		;First gpr address in bank 0
	movwf FSR		;to indirect address register.
INITB0:	clrf INDF		;Clear location pointed to.
	incf FSR,F		;Select next address.
	btfss FSR,7		;Skip if end of bank 0.
	goto INITB0		;
	movlw $A0		;First gpr address in bank 1
	movwf FSR		;to indirect address register.
INITB1:	clrf INDF		;Clear location pointed to.
	incf FSR,F		;Select next address.
	btfss STATUS,Z		;Skip if end of bank 1.
	goto INITB1		;
	BLOCK1			;
	movlw $20		;First gpr address in bank 2
	movwf FSR		;to indirect address register.
INITB2:	clrf INDF		;Clear location pointed to.
	incf FSR,F		;Select next address.
	movf FSR,W		;Skip if
	sublw $50		;end of
	btfss STATUS,Z		;bank 2.
	goto INITB2		;
	BLOCK0			;
				;Set up ports and timers.
	clrf PORTA		;Clear port.
        clrf PORTB		;Clear port.
	movlw %00000111		;Port A bits 0..3 as digital
	movwf CMCON		;input/output
        RP0SET			;Bank 1.
	movlw %00100000		;All port A bits as outputs except bit 5
	movwf TRISA		;is always input with MCLRE cleared.
	movlw %11110000		;Port B bits 0..3 outputs
        movwf TRISB      	;and bits 4..7 as inputs.
        movlw %10000011 	;Set timer 0 ratio 1:8,
        movwf OPTION		;and turn off port B weak pull-ups.
        RP0CLR			;Bank 0.
	movlw %00110001		;Set up timer 1; bit 0 enable, bit 1 Fosc/4, bit 2 X,
	movwf T1CON		;bit 3 not internal oscillator, bits 4/5 prescale 8,
				;bits 6/7 not implemented; results in overflow every
				;524.3 ms with 4 MHz clock, but see value for T1MASK.
	clrf INTCON		;Disable all interrupts.
	movf TMR1H,W		;Initialise counter show
	;movwf TMR1HS		;interval timer.
	movlw LCINIT		;Initialise lamp toggle
	movwf LMPCNT		;count down.
	
				;Divert here if doing calibration.
	btfsc PORTA,5		;Skip if not calibrating.
	goto INITI1		;Else calibrate.
	call READEEPROMDATA	;Get step count values from EEPROM.
	goto BACKGROUND		;
INITI1:	bsf PORTA,6		;Turn both lamps
	bsf PORTA,7		;on and do
	goto CALIBRATELOOP	;calibration.

;..........

		;Output lower nibb from wave table output to port A,
		;preserving the upper nibb.
O2PORTA:	movf PORTA,W	;Get port A and mask
	andlw %11110000		;in upper nibb.
	iorwf TABOUT,W		;Inclusive OR lower nibb
	movwf PORTA		;and output to port A.
	return			;

;..........

		;Output lower nibb from wave table output to port B,
		;preserving the upper nibb.
O2PORTB:	movf PORTB,W	;Get port B and mask
	andlw %11110000		;in upper nibb.
	iorwf TABOUT,W		;Inclusive OR lower nibb
	movwf PORTB		;and output to port B.
	return			;

;..........

READEEPROM: nop	;Reads data from the EEPROM at address given in W.
		;Byte read is returned in W.
		;Called from ..............; calls none.
	RP0SET			;Bank 1.
	movwf EEADR		;Set address to be accessed.
	bsf EECON1,RD		;Initiate data read.
	movf EEDATA,W		;Read data into W.
	RP0CLR			;Bank 0.
	return			;

;..........

READEEPROMDATA:	nop	;Reads the step count data from EEPROM into GPRs.
	clrf LOOPA		;Initialise counter and pointer.
RDEED1:	movfw LOOPA		;Get next value
	call READEEPROM		;from EEPROM
	movwf EEPROD		;and store it.
	movlw CTGRAH		;Point to
	addwf LOOPA,W		;next location
	movwf FSR		;to store
	movfw EEPROD		;and store
	movwf INDF		;value.
	incf LOOPA,W		;Increment
	sublw NUMSTV		;loop counter
	btfsc STATUS,Z		;and go
	return			;back
	incf LOOPA,F		;if not
	goto RDEED1		;done.	
	
;..........

		;Sets A signal to green and keeps it there.
SETA2G:	btfsc BSYFLA,SG		;Skip if busy flag clear.
	goto STA2G1		;
	btfsc SIGSTA,R		;Skip if signal state not R.
	goto STA2G2		;
	btfsc SIGSTA,Y		;Skip if signal state not Y.
	goto STA2G3		;	
	return			;
STA2G1:	call STACWA		;Step A ACW.
	call DECNTA		;Decrement count.
	addlw 255		;Test and skip
	btfss STATUS,Z		;if not done.
	goto STA2G4		;Else stop stepping.
	return			;
STA2G2:	movfw CTRGAH		;Set red
	movwf CNTAH		;to green
	movfw CTRGAL		;counter
	movwf CNTAL		;values.
	bsf BSYFLA,SG		;Set busy flag.
	return			;
STA2G3:	movfw CTYGAH		;Set yellow
	movwf CNTAH		;to green
	movfw CTYGAL		;counter
	movwf CNTAL		;values.
	bsf BSYFLA,SG		;Set busy flag.
	return			;
STA2G4:	clrf BSYFLA		;Clear busy flags.
	clrf SIGSTA		;Set signal state
	bsf SIGSTA,G		;to green.
	return			;
	
;..........

		;Sets A signal to red and keeps it there.
SETA2R:	btfsc BSYFLA,SR		;Skip if busy flag clear.
	goto STA2R1		;
	btfsc SIGSTA,Y		;Skip if signal state not Y.
	goto STA2R2		;
	btfsc SIGSTA,G		;Skip if signal state not G.
	goto STA2R3		;	
	return			;
STA2R1:	call STCWA		;Step A CW.
	call DECNTA		;Decrement count.
	addlw 255		;Test and skip
	btfss STATUS,Z		;if not done.
	goto STA2R4		;Else stop stepping.
	return			;
STA2R2:	movfw CTYRAH		;Set yellow
	movwf CNTAH		;to red
	movfw CTYRAL		;counter
	movwf CNTAL		;values.
	bsf BSYFLA,SR		;Set busy flag.
	return			;
STA2R3:	movfw CTGRAH		;Set green
	movwf CNTAH		;to red
	movfw CTGRAL		;counter
	movwf CNTAL		;values.
	bsf BSYFLA,SR		;Set busy flag.
	return			;
STA2R4:	clrf BSYFLA		;Clear busy flags.
	clrf SIGSTA		;Set signal state
	bsf SIGSTA,R		;to red.
	return			;


;..........

		;Sets A signal to yellow and keeps it there.
SETA2Y:	btfsc BSYFLA,SY		;Skip if busy flag clear.
	goto STA2Y1		;
	btfsc SIGSTA,R		;Skip if signal state not R.
	goto STA2Y4		;
	btfsc SIGSTA,G		;Skip if signal state not G.
	goto STA2Y5		;
	return			;
STA2Y1:	btfsc BSYFLA,CW		;Skip if not CW stepping.
	goto STA2Y2		;
	call STACWA		;Step A ACW.
	goto STA2Y3		;
STA2Y2:	call STCWA		;Step A CW.
STA2Y3:	call DECNTA		;Decrement count.
	addlw 255		;Test and skip
	btfss STATUS,Z		;if not done.
	goto STA2Y6		;Else stop stepping.
	return			;
STA2Y4:	movfw CTRYAH		;Set red
	movwf CNTAH		;to yellow
	movfw CTRYAL		;counter
	movwf CNTAL		;values.
	bcf BSYFLA,CW		;Set ACW rotation.
	bsf BSYFLA,SY		;Set busy flag.
	return			;
STA2Y5:	movfw CTGYAH		;Set green
	movwf CNTAH		;to yellow
	movfw CTGYAL		;counter
	movwf CNTAL		;values.
	bsf BSYFLA,CW		;Set CW rotation.
	bsf BSYFLA,SY		;Set busy flag.
	return			;
STA2Y6:	clrf BSYFLA		;Clear busy flags.
	clrf SIGSTA		;Set signal state
	bsf SIGSTA,Y		;to yellow.
	return			;

;..........

		;Sets B signal to green and keeps it there.
SETB2G:	btfsc BSYFLB,SG		;Skip if busy flag clear.
	goto STB2G1		;
	btfsc SIGSTB,R		;Skip if signal state not R.
	goto STB2G2		;
	btfsc SIGSTB,Y		;Skip if signal state not Y.
	goto STB2G3		;	
	return			;
STB2G1:	call STACWB		;Step B ACW.
	call DECNTB		;Decrement count.
	addlw 255		;Test and skip
	btfss STATUS,Z		;if not done.
	goto STB2G4		;Else stop stepping.
	return			;
STB2G2:	movfw CTRGBH		;Set red
	movwf CNTBH		;to green
	movfw CTRGBL		;counter
	movwf CNTBL		;values.
	bsf BSYFLB,SG		;Set busy flag.
	return			;
STB2G3:	movfw CTYGBH		;Set yellow
	movwf CNTBH		;to green
	movfw CTYGBL		;counter
	movwf CNTBL		;values.
	bsf BSYFLB,SG		;Set busy flag.
	return			;
STB2G4:	clrf BSYFLB		;Clear busy flags.
	clrf SIGSTB		;Set signal state
	bsf SIGSTB,G		;to green.
	return			;

;..........

		;Sets B signal to red and keeps it there.
SETB2R:	btfsc BSYFLB,SR		;Skip if busy flag clear.
	goto STB2R1		;
	btfsc SIGSTB,Y		;Skip if signal state not Y.
	goto STB2R2		;
	btfsc SIGSTB,G		;Skip if signal state not G.
	goto STB2R3		;	
	return			;
STB2R1:	call STCWB		;Step B CW.
	call DECNTB		;Decrement count.
	addlw 255		;Test and skip
	btfss STATUS,Z		;if not done.
	goto STB2R4		;Else stop stepping.
	return			;
STB2R2:	movfw CTYRBH		;Set yellow
	movwf CNTBH		;to red
	movfw CTYRBL		;counter
	movwf CNTBL		;values.
	bsf BSYFLB,SR		;Set busy flag.
	return			;
STB2R3:	movfw CTGRBH		;Set green
	movwf CNTBH		;to red
	movfw CTGRBL		;counter
	movwf CNTBL		;values.
	bsf BSYFLB,SR		;Set busy flag.
	return			;
STB2R4:	clrf BSYFLB		;Clear busy flags.
	clrf SIGSTB		;Set signal state
	bsf SIGSTB,R		;to red.
	return			;

;..........

		;Sets B signal to yellow and keeps it there.
SETB2Y:	btfsc BSYFLB,SY		;Skip if busy flag clear.
	goto STB2Y1		;
	btfsc SIGSTB,R		;Skip if signal state not R.
	goto STB2Y4		;
	btfsc SIGSTB,G		;Skip if signal state not G.
	goto STB2Y5		;
	return			;
STB2Y1:	btfsc BSYFLB,CW		;Skip if not CW stepping.
	goto STB2Y2		;
	call STACWB		;Step B ACW.
	goto STB2Y3		;
STB2Y2:	call STCWB		;Step B CW.
STB2Y3:	call DECNTB		;Decrement count.
	addlw 255		;Test and skip
	btfss STATUS,Z		;if not done.
	goto STB2Y6		;Else stop stepping.
	return			;
STB2Y4:	movfw CTRYBH		;Set red
	movwf CNTBH		;to yellow
	movfw CTRYBL		;counter
	movwf CNTBL		;values.
	bcf BSYFLB,CW		;Set ACW rotation.
	bsf BSYFLB,SY		;Set busy flag.
	return			;
STB2Y5:	movfw CTGYBH		;Set green
	movwf CNTBH		;to yellow
	movfw CTGYBL		;counter
	movwf CNTBL		;values.
	bsf BSYFLB,CW		;Set CW rotation.
	bsf BSYFLB,SY		;Set busy flag.
	return			;
STB2Y6:	clrf BSYFLB		;Clear busy flags.
	clrf SIGSTB		;Set signal state
	bsf SIGSTB,Y		;to yellow.
	return			;

;..........

		;Increment count and step once ACW for port A, output 1.
STACWA:	incf COUNTA,F		;Increment count.
	movf COUNTA,W		;Output
	call WAVE		;new bit
	movwf TABOUT		;to
	call O2PORTA		;port A.
	return			;

;..........

		;Increment count and step once ACW for port B, output 2.
STACWB:	incf COUNTB,F		;Increment count.
	movf COUNTB,W		;Output
	call WAVE		;new bit
	movwf TABOUT		;to
	call O2PORTB		;port B.
	return			;

;..........

		;Decrement count and step once CW for port A, output 1.
STCWA:	decf COUNTA,F		;Decrement count.
	movf COUNTA,W		;Output
	call WAVE		;new bit
	movwf TABOUT		;to
	call O2PORTA		;port A.
	return			;

;..........

		;Decrement count and step once CW for port B, output 2.
STCWB:	decf COUNTB,F		;Decrement count.
	movf COUNTB,W		;Output
	call WAVE		;new bit
	movwf TABOUT		;to
	call O2PORTB		;port B.
	return			;

;..........

		;Turn off both stepper motors.
STPOFF:	movf PORTA,W		;Mask out
	andlw %11110000		;low nibb
	movwf PORTA		;port A.
	movf PORTB,W		;Mask out
	andlw %11110000		;low nibb
	movwf PORTB		;port B.
	return			;
	
;..........	

		;Toggle both lamps, if lamp count is zero.
TOGLPS:	decf LMPCNT,F		;Skip if decremented lamp
	btfss STATUS,Z		;count equals zero
	return			;
	movlw LCINIT		;Reset lamp
	movwf LMPCNT		;count.
	movf PORTA,W		;Get lamp bits
	andlw %11000000		;from port A,
	xorlw %11000000		;toggle them
	movwf LMPOUT		;and store.
	movf PORTA,W		;Mask in other bits
	andlw %00111111		;from port A,
	iorwf LMPOUT,W		;fold in lamp bits
	movwf PORTA		;and output to port A.
	return			;

;..........

WRITEEEPROM:	nop	;Writes the data in EEDATA to the address in EEADR to the EEPROM.
	RP0SET			;Bank 1.
	bsf EECON1,WREN		;Enable write.
	bcf INTCON,GIE		;Disable interrupts.
	movlw $55		;Write
	movwf EECON2		;required
	movlw $AA		;
	movwf EECON2		;sequence.
	bsf EECON1,WR		;Start writing.
WREEP1:	btfsc EECON1,WR		;Skip if done writing.
	goto WREEP1		;Else go back.
	bcf EECON1,WREN		;Disable write.
	RP0CLR			;Bank 0.
	return			;
	
;..........

